<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Server;
use App\Models\News;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Cache;

class HomeController extends Controller
{
    public function index()
    {

        
        $usersCount = User::count();
        
        // Get servers and their live status
        $servers = $this->getServersWithStatus();
        
        $latestNews = News::latest()->take(3)->get();

        return view('home', [
            'users' => $usersCount,
            'servers' => $servers, 
            'latestNews' => $latestNews,
        ]);
    }

    public function users()
    {
        $users = User::paginate(30);

        return view('userlist', compact('users'));
    }
    
    private function getServersWithStatus()
    {
        $servers = Server::where('is_active', true)->get();
        
        $steamApiKey = config_value('steam_apikey');
        
        foreach ($servers as $server) {
            // Cache server status for 5 minutes to avoid excessive API calls
            $cacheKey = "server_status_{$server->id}";
            
            $serverInfo = Cache::remember($cacheKey, 300, function () use ($server, $steamApiKey) {
                return $server->queryServerInfo($steamApiKey);
            });
            
            if ($serverInfo) {
                $server->status = 'online';
                $server->current_players = $serverInfo['players'] ?? 0;
                $server->max_players = $serverInfo['max_players'] ?? 0;
                $server->current_map = $serverInfo['map'] ?? 'Unknown';
                $server->gamemode = $serverInfo['gametype'] ?? 'Unknown';
                $server->connect_url = "steam://connect/{$server->ip}:{$server->port}";
            } else {
                $server->status = 'offline';
                $server->current_players = 0;
                $server->max_players = '?';
                $server->current_map = 'Unknown';
                $server->gamemode = 'Unknown';
                $server->connect_url = "steam://connect/{$server->ip}:{$server->port}";
            }
        }
        
        return $servers;
    }

    public function privacy()
    {
        return view('privacy-policy');
    }
}